#include "mcmini_private.h"
#include "MCSharedTransition.h"
#include "MCTransitionFactory.h"
#include "signals.h"
#include "transitions/MCTransitionsShared.h"
#include <vector>
#include <sys/wait.h> // For waitpid
#include <errno.h>    // For errno
#include <cstring>    // For strerror
#include <iostream>   // For std::cerr

extern "C" {
#include "mc_shared_sem.h"
#include <cassert>
#include <cstdio>
#include <fcntl.h>
#include <signal.h>
#include <sys/mman.h>
#include <sys/prctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>
#include <ucontext.h>
}

using namespace std;

MC_THREAD_LOCAL tid_t tid_self = TID_INVALID;
pid_t trace_pid                = -1;

trid_t traceId      = 0;
trid_t transitionId = 0;

time_t mcmini_start_time = 0;
volatile bool mc_reset = false;

/**
 * The process id of the scheduler
 */
pid_t scheduler_pid = -1;
mc_shared_sem (*trace_sleep_list)[MAX_TOTAL_THREADS_IN_PROGRAM] =
  nullptr;
sem_t mc_pthread_create_binary_sem;

static char resultString[1000] = "***** Model checking completed! *****\n";
static void addResult(const char *result) {
  char stats[1000];
  if (strstr(resultString, result) != NULL) {
    result = "  (Other trace numbers (traceId) of bugs exist above;\n"
             "   Use --first-deadlock (-f) to only show first one)\n";
    if (strstr(resultString, result) == NULL) {
      strncat(resultString, result, sizeof(resultString) - strlen(resultString));
    }
    return;
  }
  strncat(resultString, result, sizeof(resultString) - strlen(resultString));
  snprintf(stats, 80, "  (Trace number (traceId): %lu)\n", traceId);
  strncat(resultString, stats, sizeof(resultString) - strlen(resultString));
}
static void printResults() {
  if (strcmp(resultString, "***** Model checking completed! *****\n") == 0) {
    // If we never previously added a conclusion, then this is the default.
    strncat(resultString, "      (NO FAILURE DETECTED)\n",
            sizeof(resultString) - 80);
  }
  mcprintf(resultString);
  mcprintf("Number of traces: %lu\n", traceId);
  mcprintf("Total number of transitions: %lu\n", transitionId);
  mcprintf("Elapsed time: %lu seconds\n", time(NULL) - mcmini_start_time);
  if ((int)traceId < programState->traceIdForPrintBacktrace() &&
      getenv(ENV_FIRST_DEADLOCK) == NULL) { // and no --first-deadlock
    mcprintf("*** NOTE: --trace (-t) requested up to trace %d,\n"
            "           but total number of traces was only %d.\n",
            programState->traceIdForPrintBacktrace(), traceId);
  }
}

/*
 * Dynamically updated to control how McMini proceeds with its
 * execution.
 *
 * If at any point during the (re-)execution of the trace McMini
 * realizes that the current trace is no longer interesting it will
 * continue searching new traces as before until either the state
 * space has been exhausted or possibly McMini stops again to
 * re-execute a future trace
 *
 * NOTE: Misusing this variable could cause all sorts of bad stuff to
 * happen. McMini relies on this value to determine if it should stop
 * execution in the middle of regenerating a trace.
 *
 * TODO: We should perhaps figure out a way to absorb this information
 * into the debugger, or perhaps make it a local variable.
 * Effectively, we should make it more difficult to set this value
 * since it affects the behavior of a rather important function for
 * state regeneration (viz. mc_fork_next_trace_at_current_state())
 */
/* Data transfer */
void *shmStart                            = nullptr;
MCSharedTransition *shmTransitionTypeInfo = nullptr;
void *shmTransitionData                   = nullptr;
const size_t shmAllocationSize =
  sizeof(*trace_sleep_list) +
  (sizeof(*shmTransitionTypeInfo) + MAX_SHARED_MEMORY_ALLOCATION);

/* Program state */
MCDeferred<MCStack> programState;

void
alarm_handler(int sig)
{
  if (sig == SIGALRM) {
    fprintf(stderr,
            "\n *** mcmini exiting after one hour.  To avoid, this,\n"
            " *** Use flag '--long-test' or  MCMINI_LONG_TEXT env. "
            "var.\n\n");
    _exit(1); // Note:  McMini wraps 'exit()'.  So, we use '_exit()'.
  }
}

ucontext_t mcmini_scheduler_main_context;

MC_CONSTRUCTOR void
mcmini_main()
{
  mcmini_start_time = time(NULL);

  getcontext(&mcmini_scheduler_main_context);

  if (getenv("MCMINI_PROCESS") == NULL) {
    setenv("MCMINI_PROCESS", "SCHEDULER", 1); // This is McMini scheduler proc
  } else { // Else this is McMini target process
    unsetenv("MCMINI_PROCESS");
    // Avoid polluting environment of target.  Or can set to "TARGET".
    return;
  }

  if (getenv(ENV_LONG_TEST) == NULL) {
    alarm(3600); // one hour
    signal(SIGALRM, alarm_handler);
  }
  mc_load_intercepted_symbol_addresses();
  mc_create_global_state_object();
  mc_initialize_shared_memory_globals();
  mc_initialize_trace_sleep_list();
  install_sighandles_for_scheduler();

  // Mark this process as the scheduler
  scheduler_pid = getpid();
  MC_FATAL_ON_FAIL(
    __real_sem_init(&mc_pthread_create_binary_sem, 0, 0) == 0);

  mc_do_model_checking();

  printResults();
  mc_stop_model_checking(EXIT_SUCCESS);
}

void
mc_create_global_state_object()
{
  auto config = get_config_for_execution_environment();
  programState.Construct(config);
  programState->registerVisibleOperationType(typeid(MCThreadStart),
                                             &MCReadThreadStart);
  programState->registerVisibleOperationType(typeid(MCThreadCreate),
                                             &MCReadThreadCreate);
  programState->registerVisibleOperationType(typeid(MCThreadFinish),
                                             &MCReadThreadFinish);
  programState->registerVisibleOperationType(typeid(MCThreadJoin),
                                             &MCReadThreadJoin);
  programState->registerVisibleOperationType(typeid(MCMutexInit),
                                             &MCReadMutexInit);
  programState->registerVisibleOperationType(typeid(MCMutexUnlock),
                                             &MCReadMutexUnlock);
  programState->registerVisibleOperationType(typeid(MCMutexLock),
                                             &MCReadMutexLock);
  programState->registerVisibleOperationType(typeid(MCSemInit),
                                             &MCReadSemInit);
  programState->registerVisibleOperationType(typeid(MCSemPost),
                                             &MCReadSemPost);
  programState->registerVisibleOperationType(typeid(MCSemWait),
                                             &MCReadSemWait);
  programState->registerVisibleOperationType(typeid(MCSemEnqueue),
                                             &MCReadSemEnqueue);
  programState->registerVisibleOperationType(typeid(MCExitTransition),
                                             &MCReadExitTransition);
  programState->registerVisibleOperationType(
    typeid(MCAbortTransition), &MCReadAbortTransition);
  programState->registerVisibleOperationType(typeid(MCBarrierEnqueue),
                                             &MCReadBarrierEnqueue);
  programState->registerVisibleOperationType(typeid(MCBarrierInit),
                                             &MCReadBarrierInit);
  programState->registerVisibleOperationType(typeid(MCBarrierWait),
                                             &MCReadBarrierWait);
  programState->registerVisibleOperationType(typeid(MCCondInit),
                                             &MCReadCondInit);
  programState->registerVisibleOperationType(typeid(MCCondSignal),
                                             &MCReadCondSignal);
  programState->registerVisibleOperationType(typeid(MCCondBroadcast),
                                             &MCReadCondBroadcast);
  programState->registerVisibleOperationType(typeid(MCCondWait),
                                             &MCReadCondWait);
  programState->registerVisibleOperationType(typeid(MCCondEnqueue),
                                             &MCReadCondEnqueue);
  programState->registerVisibleOperationType(typeid(MCRWLockInit),
                                             &MCReadRWLockInit);
  programState->registerVisibleOperationType(
    typeid(MCRWLockReaderEnqueue), &MCReadRWLockReaderEnqueue);
  programState->registerVisibleOperationType(
    typeid(MCRWLockWriterEnqueue), &MCReadRWLockWriterEnqueue);
  programState->registerVisibleOperationType(
    typeid(MCRWLockWriterLock), &MCReadRWLockWriterLock);
  programState->registerVisibleOperationType(
    typeid(MCRWLockReaderLock), &MCReadRWLockReaderLock);
  programState->registerVisibleOperationType(typeid(MCRWLockUnlock),
                                             &MCReadRWLockUnlock);

  programState->registerVisibleOperationType(typeid(MCRWWLockInit),
                                             &MCReadRWWLockInit);
  programState->registerVisibleOperationType(
    typeid(MCRWWLockReaderEnqueue), &MCReadRWWLockReaderEnqueue);
  programState->registerVisibleOperationType(
    typeid(MCRWWLockReaderLock), &MCReadRWWLockReaderLock);
  programState->registerVisibleOperationType(
    typeid(MCRWWLockWriter1Enqueue), &MCReadRWWLockWriter1Enqueue);
  programState->registerVisibleOperationType(
    typeid(MCRWWLockWriter1Lock), &MCReadRWWLockWriter1Lock);
  programState->registerVisibleOperationType(
    typeid(MCRWWLockWriter2Enqueue), &MCReadRWWLockWriter2Enqueue);
  programState->registerVisibleOperationType(
    typeid(MCRWWLockWriter2Lock), &MCReadRWWLockWriter2Lock);
  programState->registerVisibleOperationType(typeid(MCRWWLockUnlock),
                                             &MCReadRWWLockUnlock);
  programState->registerVisibleOperationType(
    typeid(MCGlobalVariableRead), &MCReadGlobalRead);
  programState->registerVisibleOperationType(
    typeid(MCGlobalVariableWrite), &MCReadGlobalWrite);
  programState->start();
}

int countVisibleObjectsOfType(int objectId) {
  int count = 0;
  for (int i = 0; i <= objectId; i++) {
    if (typeid(*(programState->getObjectWithId(i))) ==
        typeid(*(programState->getObjectWithId(objectId)))) {
      count++;
    }
  }
  return count;
}

void
mc_prepare_to_model_check_new_program()
{
  mc_register_main_thread();
  auto mainThread = programState->getThreadWithId(TID_MAIN_THREAD);
  auto initialTransition =
    MCTransitionFactory::createInitialTransitionForThread(mainThread);
  programState->setNextTransitionForThread(TID_MAIN_THREAD,
                                           initialTransition);
}

int
mc_explore_branch(int curBranchPoint)
{
  tid_t backtrackThread;

  if (curBranchPoint == FIRST_BRANCH) {
    mc_fork_new_trace();
    backtrackThread = TID_MAIN_THREAD;
  } else { // else next branch
    auto *sNext = &(programState->getStateItemAtIndex(curBranchPoint));
    backtrackThread = sNext->popThreadToBacktrackOn();

    // Prepare the scheduler's model of the next trace
    programState->reflectStateAtTransitionIndex(curBranchPoint - 1);

    mc_fork_next_trace_at_current_state();
  }

  mc_search_dpor_branch_with_thread(backtrackThread);
  // If '-t <traceId>' set and current traceId matches it, then exit.
  mc_exit_with_trace_if_necessary(traceId);

  traceId++;
  if (false && traceId >= 1 && getenv(ENV_PRINT_AT_TRACE_SEQ) != NULL) {
    mcprintf("*** Trace sequence ('-t', --trace') requested.\n"
             "*** for more than one traceId: -t<X> -t'<traceSeq>' for X>0\n"
             "*** McMini cannot yet handle this situation.  Exiting now.\n");
    mc_exit(EXIT_FAILURE);
  }
  resetTraceSeqArray();

  static time_t last_time_reported = mcmini_start_time;
  static int interval = 1000;
  if (traceId == 100 && time(NULL) - last_time_reported > 10) {
    interval = 100;
  }
  if (traceId % interval == 0) {
    if (time(NULL) - last_time_reported > 10) {
      last_time_reported = time(NULL);
      mcprintf("... %d traces analyzed so far ...\n", traceId);
    }
  }
  return programState->getDeepestDPORBranchPoint();
}

bool
isInLivelock()
{
  const MCTransition *nextTransition = programState->getFirstEnabledTransition();
  bool hasLivelock = false;
  MCTransitionUniqueRep
  traceArr[MC_STATE_CONFIG_MAX_TRANSITIONS_DEPTH_LIMIT_DEFAULT] = {};
  int traceLen = 0;

  while (nextTransition != nullptr && transitionId < MC_STATE_CONFIG_MAX_TRANSITIONS_DEPTH_LIMIT_DEFAULT) {
    tid_t tid = nextTransition->getThreadId();
    mc_run_thread_to_next_visible_operation(tid);


    #ifdef DEBUG
      mcprintf("Program State: ");
      programState->printDebugProgramState();
      nextTransition->print();
    #endif

    programState->simulateRunningTransition(
      *nextTransition, shmTransitionTypeInfo, shmTransitionData);

    if (getenv(ENV_CHECK_FOR_WEAK_LIVELOCK)) {
      nextTransition = programState->getFirstEnabledTransition();
    }
    else {
      nextTransition = programState->getNextFairTransition(tid);
    }
    transitionId++;
  }

  if (nextTransition == nullptr && !programState->isInDeadlock()) {
    programState->copyCurrentTraceToArray(traceArr, traceLen);
    // FUTURE EXTENSION: User declares progress; no livelock 
    // hasLivelock = !programState->isProgress(traceArr, traceLen);
    hasLivelock = programState->hasRepetition(traceArr, traceLen);
  }
  else if (transitionId >= MAX_TOTAL_TRANSITIONS_IN_PROGRAM) {
    printResults();
    mcprintf(
      "*** Execution Limit Reached! ***\n\n"
      "McMini ran a trace with %lu transitions.  To increase this limit,\n"
      "modify MAX_TOTAL_TRANSITIONS_IN_PROGRAM in MCConstants.h and"
      " re-compile.\n"
      "But first, try running mcmini with the \"--max-depth-per-thread\""
      " flag (\"-m\")\n"
      "to limit how far into a trace a McMini thread can go.\n",
      transitionId);
    mc_stop_model_checking(EXIT_FAILURE);
  }
  return hasLivelock;
}

void
mc_do_model_checking()
{
  mc_prepare_to_model_check_new_program();

  int nextBranchPoint = mc_explore_branch(FIRST_BRANCH);
  while (nextBranchPoint != FIRST_BRANCH) { // while not backtracked to origin
    nextBranchPoint = mc_explore_branch(nextBranchPoint);
    nextBranchPoint = programState->getDeepestDPORBranchPoint();
  }
}

void
mc_get_shm_handle_name(char *dst, size_t sz)
{
  snprintf(dst, sz, "/mcmini-%s-%lu", getenv("USER"), (long)getpid());
  dst[sz - 1] = '\0';
}

void *
mc_allocate_shared_memory_region()
{
  //  If the region exists, then this returns a fd for the existing
  //  region. Otherwise, it creates a new shared memory region.
  char dpor[100];
  mc_get_shm_handle_name(dpor, sizeof(dpor));

  // This creates a file in /dev/shm/
  int fd = shm_open(dpor, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR);
  if (fd == -1) {
    if (errno == EACCES) {
      fprintf(stderr,
              "Shared memory region '%s' not owned by this process\n",
              dpor);
    } else {
      perror("shm_open");
    }
    mc_exit(EXIT_FAILURE);
  }
  int rc = ftruncate(fd, shmAllocationSize);
  if (rc == -1) {
    perror("ftruncate");
    mc_exit(EXIT_FAILURE);
  }
  // We want stack at same address for each process.  Otherwise, a
  // pointer
  //   to an address in the stack data structure will not work
  //   everywhere. Hopefully, this address is not already used.
  void *stack_address = (void *)0x4444000;
  void *shmStart =
    mmap(stack_address, shmAllocationSize, PROT_READ | PROT_WRITE,
         MAP_SHARED | MAP_FIXED, fd, 0);
  if (shmStart == MAP_FAILED) {
    perror("mmap");
    mc_exit(EXIT_FAILURE);
  }
  // shm_unlink(dpor); // Don't unlink while child processes need to
  // open this.
  fsync(fd);
  close(fd);
  return shmStart;
}

void
mc_deallocate_shared_memory_region()
{
  char shm_file_name[100];
  mc_get_shm_handle_name(shm_file_name, sizeof(shm_file_name));
  int rc = munmap(shmStart, shmAllocationSize);
  if (rc == -1) {
    perror("munmap");
    mc_exit(EXIT_FAILURE);
  }

  rc = shm_unlink(shm_file_name);
  if (rc == -1) {
    if (errno == EACCES) {
      fprintf(stderr,
              "Shared memory region '%s' not owned by this process\n",
              shm_file_name);
    } else {
      perror("shm_unlink");
    }
    mc_exit(EXIT_FAILURE);
  }
}

void
mc_initialize_shared_memory_globals()
{
  void *shm              = mc_allocate_shared_memory_region();
  void *threadQueueStart = shm;
  void *shmTransitionTypeInfoStart =
    (char *)threadQueueStart + sizeof(*trace_sleep_list);
  void *shmTransitionDataStart = (char *)shmTransitionTypeInfoStart +
                                 sizeof(*shmTransitionTypeInfo);

  shmStart = shm;
  trace_sleep_list =
    static_cast<typeof(trace_sleep_list)>(threadQueueStart);
  shmTransitionTypeInfo = static_cast<typeof(shmTransitionTypeInfo)>(
    shmTransitionTypeInfoStart);
  shmTransitionData = shmTransitionDataStart;
}

void
mc_initialize_trace_sleep_list()
{
  for (unsigned int i = 0; i < MAX_TOTAL_THREADS_IN_PROGRAM; i++)
    mc_shared_sem_init(&(*trace_sleep_list)[i]);
}

void
mc_reset_cv_locks()
{
  for (unsigned int i = 0; i < MAX_TOTAL_THREADS_IN_PROGRAM; i++) {
    mc_shared_sem_destroy(&(*trace_sleep_list)[i]);
    mc_shared_sem_init(&(*trace_sleep_list)[i]);
  }
}

void
mc_fork_new_trace()
{
  // Ensure that a child does not already
  // exist to prevent fork bombing
  MC_ASSERT(trace_pid == -1);

  pid_t childpid;
  if ((childpid = fork()) < 0) {
    perror("fork");
    abort();
  }
  trace_pid = childpid;

  if (FORK_IS_CHILD_PID(childpid)) {
    prctl(PR_SET_PDEATHSIG, SIGUSR1, 0, 0); // In McMini, SIGUSR1 to kill child
    if (getenv(ENV_QUIET) != NULL) {
      close(0); assert(open("/dev/null", O_RDONLY) == 0);
      close(1); assert(open("/dev/null", O_WRONLY) == 1);
      close(2); assert(open("/dev/null", O_WRONLY) == 2);
    }

    install_sighandles_for_trace();

    // We need to reset the concurrent system
    // for the child since, at the time this method
    // is invoked, it will have a complete copy of
    // the state the of system. But we need to
    // re-simulate the system by running the transitions
    // in the transition stack; otherwise, shadow resource
    // allocations will be off
    programState->reset();
    programState->start();
    mc_register_main_thread();

    // NOTE: Technically, the child will be frozen
    // inside of dpor_init until it is scheduled. But
    // this is only a technicality: it doesn't actually
    // matter where the child spawns so long as it reaches
    // the actual source program
    tid_self = 0;

    // Note that the child process does
    // not need to re-map the shared memory
    // region as the parent has already done that

    // This is important to handle the case when the
    // main thread hits return 0; in that case, we
    // keep the process alive to allow the model checker to
    // continue working
    //
    // NOTE!!: atexit handlers can be invoked when a dynamic
    // library is unloaded. In the transparent target, we need
    // to be able to handle this case gracefully
    MC_FATAL_ON_FAIL(atexit(&mc_exit_main_thread) == 0);

    thread_await_scheduler_for_thread_start_transition();

    setcontext(&mcmini_scheduler_main_context);
  }
}

void
mc_fork_next_trace_at_current_state()
{
  mc_reset_cv_locks();
  mc_fork_new_trace();

  const int tStackHeight = programState->getTransitionStackSize();

  for (int i = 0; i < tStackHeight; i++) {
    // NOTE: This is reliant on the fact
    // that threads are created in the same order
    // when we create them. This will always be consistent,
    // but we might need to look out for when a thread dies
    tid_t nextTid = programState->getThreadRunningTransitionAtIndex(i);
    mc_run_thread_to_next_visible_operation(nextTid);
  }
}

void mc_run_thread_to_next_visible_operation(tid_t tid) {
  MC_ASSERT(tid != TID_INVALID);
  mc_shared_sem_ref sem = &(*trace_sleep_list)[tid];
  // Slightly dangerous: We're depending on sem FIFO policy.
  // We post to sem.  Then tid wakes up and runs while we wait on sem.
  // Then tid reaches next visible operation, posts to us, and waits.
  // But suppose we post to tid, we wait, and wakeup goes to us, not to tid.
  // A more careful version would use two semaphores: "tid wait" and "we wait".
  mc_shared_sem_wake_thread(sem);
  mc_shared_sem_wait_for_thread(sem);
}

void mc_terminate_trace() {
  if (mc_reset) return;  // User decided to do 'mcmini back'
  if (trace_pid == -1) return;  // No child
  kill(trace_pid, SIGUSR1);
  mc_wait_for_trace();
  trace_pid = -1;
}

void mc_wait_for_trace() {
  MC_ASSERT(trace_pid != -1);

  int status;
  char *v = getenv(ENV_VERBOSE);
  bool verbose = v ? v[0] == '1' : false;
  if (waitpid(trace_pid, &status, 0) == -1) {
    if (verbose) {
      fprintf(stderr, "Error waiting for trace process with pid `%lu` %s\n",
              (uint64_t)trace_pid, strerror(errno));
    }
  } else if (verbose) {
    // Check how the trace process exited
    if (WIFEXITED(status)) {
      fprintf(stderr,
              "Trace process with traceId `%lu` exited with status %d\n",
              traceId, WEXITSTATUS(status));
    } else if (WIFSIGNALED(status)) {
      fprintf(stderr,
              "Trace process with traceId `%lu` was killed by signal `%d`\n",
              traceId, WTERMSIG(status));
    } else {
      fprintf(stderr, "Trace process with traceId `%lu` exited abnormally.\n",
              traceId);
    }
  }
}

void mc_trace_panic() {
  pid_t schedpid = getppid();
  kill(schedpid, SIGUSR1);

  // The scheduler will kill the child
  // process before being able to leave this function
  waitpid(schedpid, nullptr, 0);
}

void mc_restore_initial_trace() {
  programState->restoreInitialTrace();
  mc_terminate_trace();
  mc_fork_next_trace_at_current_state();
}

void
mc_search_dpor_branch_with_thread(const tid_t backtrackThread)
{
  int depth = programState->getTransitionStackSize();
  if (traceId >= 1 && getenv(ENV_PRINT_AT_TRACE_SEQ) != NULL) {
    if (depth < traceSeqLength()) {
      printResults();
      mc_stop_model_checking(EXIT_SUCCESS); // Exit McMini
    } else {
      setEndOfTraceSeq(); // Stop using traceSeq[].
    }
  }
  const MCTransition &initialTransition =
    programState->getNextTransitionForThread(backtrackThread);
  const MCTransition *nextTransition = &initialTransition;

  // TODO: Assert whether or not nextTransition is enabled
  // TODO: Assert whether a trace process exists at this point

  do {
    if (transitionId >= MC_STATE_CONFIG_MAX_TRANSITIONS_DEPTH_LIMIT_DEFAULT) {
      printResults();
      mcprintf(
        "*** Execution Limit Reached! ***\n\n"
        "McMini ran a trace with %lu transitions.  To increase this limit,\n"
        "modify MAX_TOTAL_TRANSITIONS_IN_PROGRAM in MCConstants.h and"
        " re-compile.\n"
        "But first, try running mcmini with the \"--max-depth-per-thread\""
        " flag (\"-m\")\n"
        "to limit how far into a trace a McMini thread can go.\n",
        depth);
      mc_stop_model_checking(EXIT_FAILURE);
    }

    depth++;
    transitionId++;

    const tid_t tid = nextTransition->getThreadId();
    // Execute in target application
    mc_run_thread_to_next_visible_operation(tid);

    // Execute model ("simulate" transition means to update model w/ transition)
    programState->simulateRunningTransition(
      *nextTransition, shmTransitionTypeInfo, shmTransitionData);
    // Record the transition in the "history", for later backtracking
    programState->dynamicallyUpdateBacktrackSets();

    /* Check for data races */
    {
      const MCTransition &nextTransitionForTid =
        programState->getNextTransitionForThread(tid);
      if (programState->hasADataRaceWithNewTransition(
            nextTransitionForTid)) {
        mcprintf("*** DATA RACE DETECTED ***\n");
        programState->printTransitionStack();
        programState->printNextTransitions();
        addResult("*** DATA RACE DETECTED"
                  " (see pending READ/WRITE operations) ***\n");
        if (getenv(ENV_FIRST_DEADLOCK)) {
          traceId++;
          printResults();
          mc_exit(EXIT_SUCCESS);
        }
      }
    }

    nextTransition = programState->getFirstEnabledTransition();

    if (nextTransition == nullptr ||
        (traceSeqLength() > 0 && programState->isInDeadlock())) {
      bool hasDeadlock = programState->isInDeadlock();
      bool hasLivelock = 0;
      if (hasDeadlock && nextTransition != nullptr) { // Stop using traceSeq
        addResult("  [Truncating traceSeq from '-t', due to deadlock!]\n");
        nextTransition = nullptr;
      }
      if (getenv(ENV_CHECK_FOR_LIVELOCK) ||
          getenv(ENV_CHECK_FOR_WEAK_LIVELOCK) && nextTransition == nullptr) {
        programState->increaseMaxTransitionsDepthLimit(
          LLOCK_INCREASED_MAX_TRANSITIONS_DEPTH);
        hasLivelock = isInLivelock();
        /*
         * isInLivelock() exits before reaching
         * LLOCK_INCREASED_MAX_TRANSITIONS_DEPTH
         * if a deadlock was discovered.
         */
        hasDeadlock = programState->isInDeadlock();
      }
      const bool programHasNoErrors = !hasDeadlock && !hasLivelock;
      char *v = getenv(ENV_VERBOSE);
      int verbose = v ? v[0] - '0' : 0;

      if (hasDeadlock) {
        mcprintf("TraceId %lu, *** DEADLOCK DETECTED ***\n", traceId);
        programState->printTransitionStack();
        programState->printNextTransitions();
        addResult("*** DEADLOCK DETECTED ***\n");
        if (verbose) {
          mcprintf("TraceId %ld:  ", traceId);
          programState->printThreadSchedule();
        }
        if (getenv(ENV_FIRST_DEADLOCK) != NULL) {
          traceId++; // Verify "Number of traces" in printResults() is correct.
          printResults();
          mc_exit(EXIT_SUCCESS); // Exit McMini
        }
      }

      if (hasLivelock) {
        mcprintf("TraceId %lu, *** POTENTIAL LIVELOCK DETECTED ***\n", traceId);
        programState->printNextTransitions();
        addResult("*** POTENTIAL LIVELOCK DETECTED ***\n");
        if (verbose) {
          mcprintf("TraceId %ld:  ", traceId);
          programState->printThreadSchedule();
          programState->printTransitionStack();
        }
        if (getenv(ENV_FIRST_DEADLOCK) != NULL) {
          traceId++; // Verify "Number of traces" in printResults() is correct.
          printResults();
          mc_exit(EXIT_SUCCESS); // Exit McMini
        }
      }

      if (programHasNoErrors) {
        switch (verbose) {
        case 0:
          break;
        case 1:
          mcprintf("TraceId %ld:  ", traceId);
          programState->printThreadSchedule();
          break;
        case 2:
          mcprintf("TraceId: %d, *** NO FAILURE DETECTED ***\n", traceId);
          programState->printTransitionStack();
          programState->printNextTransitions();
          break;
        default:
          mcprintf("McMini: Internal error: verbose is %d\n", verbose);
          exit(1);
        }
      }

      // Three cases:
      //  1. Continue trace (nextTransition != nullptr)
      //  2. Reset trace (mcmini back set mc_reset: mc_restore_initial_trace)
      //  3. End of trace; start the next traceId (nextTransition == nullptr)
      // GDB stops in mc_wait_for_Trace (waitpid) inside mc_terminate_trace().
      mc_terminate_trace(); // Does nothing if mc_reset is true
    } // End of 'nextTransition == nullptr'; If 'mc_reset', then nextTransition

    // Let's give one more chance for 'mcmini back' to set mc_reset and go back.
    if (mc_reset) {
      // Avoid infinite loop in: mc_restore_*()->mc_terminate_trace()->mc_reset
      mc_reset = false;
      mc_restore_initial_trace();
      depth = 0;
      transitionId = 0;
      nextTransition = programState->getFirstEnabledTransition(); // Reset trace
    }

  } while (nextTransition != nullptr);
}

tid_t
mc_register_thread()
{
  tid_t newTid = programState->createNewThread();
  tid_self     = newTid;
  return newTid;
}

tid_t
mc_register_main_thread()
{
  tid_t newTid = programState->createMainThread();
  tid_self     = newTid;
  return newTid;
}

void
mc_report_undefined_behavior(const char *msg)
{
  mc_terminate_trace();
  fprintf(stderr,
          "\n"
          "Undefined or Illegal Behavior Detected!\n"
          "............................\n"
          "mcmini aborted the execution of trace with traceId %lu because\n"
          "it detected undefined behavior\n"
          "............................ \n"
          "Reason: %s\n\n",
          traceId, msg);
  programState->printTransitionStack();
  programState->printNextTransitions();
  mc_exit(EXIT_FAILURE);
}

/* GDB Interface */

void
mc_exit_with_trace_if_necessary(trid_t trid)
{
  if (programState->isTargetTraceIdForPrintBacktrace(trid)) {
    mcprintf("*** -t or --trace requested.  Printing trace:\n");
    programState->printTransitionStack();
    programState->printNextTransitions();
    traceId++; // We stopped at -t<trid>, but Number of traces shuld be trid+1.
    printResults();
    mc_stop_model_checking(EXIT_SUCCESS);
  }
}

MCStackConfiguration
get_config_for_execution_environment()
{
  // FIXME: This is also in a bad spot. This needs to removed/changed
  // completely. We shouldn't need to pass arguments through the
  // environment. This suggests that mcmini would be better as a
  // single process that forks, exec()s w/LD_PRELOAD set, and then
  // remotely controls THAT process. We need to discuss this
  uint64_t maxThreadDepth = MC_STATE_CONFIG_MAX_DEPTH_PER_THREAD_DEFAULT;
  uint64_t maxTotalDepth = MC_STATE_CONFIG_MAX_TRANSITIONS_DEPTH_LIMIT_DEFAULT - 1;
  
  if (getenv(ENV_CHECK_FOR_LIVELOCK)) {
    maxTotalDepth -= LLOCK_INCREASED_MAX_TRANSITIONS_DEPTH;
  }  

  trid_t printBacktraceAtTraceNumber = MC_STATE_CONFIG_PRINT_AT_TRACE;
  bool firstDeadlock                  = false;
  bool expectForwardProgressOfThreads = false;

  // TODO: Sanitize arguments (check errors of strtoul)
  if (getenv(ENV_MAX_DEPTH_PER_THREAD) != NULL) {
    maxThreadDepth = strtoul(getenv(ENV_MAX_DEPTH_PER_THREAD), nullptr, 10);
  }

  if (getenv(ENV_MAX_TRANSITIONS_DEPTH_LIMIT) != NULL) {
    int limit = maxTotalDepth;
    maxTotalDepth = strtoul(getenv(ENV_MAX_TRANSITIONS_DEPTH_LIMIT), nullptr, 10);
    if (maxTotalDepth >= limit) {
      maxTotalDepth = limit;
      mcprintf("\nWarning: Value of -M set to a default maximum of %d.\n"
               "(further reduced by %d when using the -l flag)\n"
               "To increase this limit, modify\n"
               "MC_STATE_CONFIG_MAX_TRANSITIONS_DEPTH_LIMIT_DEFAULT in\n"
               "MCConstants.h and re-compile.\n\n"
               "Continuing with -M = %d\n\n",
               MC_STATE_CONFIG_MAX_TRANSITIONS_DEPTH_LIMIT_DEFAULT,
               LLOCK_INCREASED_MAX_TRANSITIONS_DEPTH,
               limit);
    }
  }

  if (getenv(ENV_PRINT_AT_TRACE_ID) != NULL) {
    printBacktraceAtTraceNumber =
      strtoul(getenv(ENV_PRINT_AT_TRACE_ID), nullptr, 10);
  }
  if (getenv(ENV_CHECK_FORWARD_PROGRESS) != NULL) {
    expectForwardProgressOfThreads = true;
  }

  if (getenv(ENV_FIRST_DEADLOCK) != NULL) {
    firstDeadlock = true;
  }

  return {maxThreadDepth, maxTotalDepth, printBacktraceAtTraceNumber, firstDeadlock,
          expectForwardProgressOfThreads};
}

bool
mc_is_scheduler()
{
  return scheduler_pid == getpid();
}

void
mc_exit(int status)
{
  // The exit() function is intercepted. Calling exit() directly
  // results in a deadlock since the thread calling it will block
  // forever (McMini does not let a process exit() during model
  // checking). Keep this in mind before switching this call to
  // a different exit function
  _Exit(status);
}

void
mc_stop_model_checking(int status)
{
  mc_deallocate_shared_memory_region();
  mc_terminate_trace();
  mc_exit(status);
}
